/***************************************************************************
 *   Copyright (C) 2007 by A.J. Tavakoli                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PLANE_H
#define PLANE_H

#include <cmath>
#include <cstddef>

#include "Vector3.h"
#include "Point3.h"
#include "AABB.h"
#include "numeric.h"


namespace Math3D {


template<class Real>
class Plane {
public:
  Plane();
  Plane(const Vector3<Real>& _n, Real _d);
  Plane(const Vector3<Real>& _n, const Point3<Real>& p);

  inline const Vector3<Real>& getNormal() const { return n; }
  inline Real getOffset() const { return d; }

  inline Real distTo(const Point3<Real> &pt) const;

  inline int classifySphere(const Point3<Real> &c, Real r, Real *dist = NULL) const;
  inline int classifyAABB(const AABB<Real> &box, Real *dist = NULL) const;
private:
  Vector3<Real> n;
  Real d;
};


template<class Real>
Plane<Real>::Plane():
     n( Real(0.0), Real(0.0), Real(0.0) ),
     d( Real(0.0) ) {

}


template<class Real>
Plane<Real>::Plane(const Vector3<Real>& _n, Real _d) {
  // normalize normal vector and plane distance (just in case caller didn't)
  /*
  Real mag = _n.mag();
  n = _n/mag;
  d = _d/mag;
  */
  // for the sake of speed...
  n = _n;
  d = _d;
}


template<class Real>
Plane<Real>::Plane(const Vector3<Real>& _n, const Point3<Real>& p) {
  n = _n;
  n.normalize();
  d = n*p;
}


// returns shortest distance from point to plane
template<class Real>
inline Real Plane<Real>::distTo(const Point3<Real> &pt) const {
  return n*pt - d;
}


// classifies a sphere with center c and radius r
// (-1 = in back, 0 = intersects, 1 = in front) 
template<class Real>
inline int Plane<Real>::classifySphere(const Point3<Real> &c, Real r, Real *dist) const {
  // calculate distance from sphere to plane
  Real _dist = n*c - d;

  // store distance from sphere to plane in dist, if dist non-NULL
  if ( dist != NULL )
      (*dist) = _dist - r;

  // in front of plane?
  if ( _dist > r )
    return 1;

  // in back of plane?
  if ( _dist < -r )
    return -1;

  // intersects plane
  return 0;
}


// classifies axis-aligned bounding-box relative to a plane
// (-1 = in back, 0 = intersects, 1 = in front)
template<class Real>
inline int Plane<Real>::classifyAABB(const AABB<Real> &box, Real *dist) const {
  const Vector3<Real> &ext = box.getExt();
  Real r = ext[0]*rabs(n[0]) + ext[1]*rabs(n[1]) + ext[2]*rabs(n[2]);
  Real _dist = n*box.getCenter() - d;

  if ( dist != NULL )
    (*dist) = _dist - r;

  // behind plane?
  if ( _dist < -r )
    return -1;

  // in front of plane?
  if ( _dist > r )
    return 1;

  // intersecting
  return 0;
}


} // namespace Math3D


#endif
