/***************************************************************************
 *   Copyright (C) 2007 by A.J. Tavakoli                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "TriangleMeshEntity.h"


TriangleMeshEntity::TriangleMeshEntity(const TriangleMesh *_mesh,
                                       int _texID,
                                       int _nmID,
                                       bool _cleanUp):
     Entity(_cleanUp),
     mesh(_mesh),
     texID(_texID),
     normalMapID(_nmID) {
  computeAABB();
}


TriangleMeshEntity::~TriangleMeshEntity() {

}


void TriangleMeshEntity::computeAABB() {
  if ( NULL == mesh )
    return;

  const Video::Vertex *verts = mesh->getIndexedVerts();
  const unsigned int *indices = mesh->getIndices();
  int numIndices = mesh->getNumTris()*3;

  if ( NULL == verts || NULL == indices || numIndices <= 0 )
    return;

  Point3 minP = verts[0].pos,
         maxP = verts[0].pos;

  for ( int i=0; i < numIndices; i++ ) {
    const Point3 &v = verts[ indices[i] ].pos;
    if ( v[0] < minP[0] )
      minP[0] = v[0];
    if ( v[1] < minP[1] )
      minP[1] = v[1];
    if ( v[2] < minP[2] )
      minP[2] = v[2];

    if ( v[0] > maxP[0] )
      maxP[0] = v[0];
    if ( v[1] > maxP[1] )
      maxP[1] = v[1];
    if ( v[2] > maxP[2] )
      maxP[2] = v[2];
  }

  aabbExt = (maxP - minP)*0.5f;
}


void TriangleMeshEntity::render() {
  if ( NULL == mesh )
    return;

  bool usingNormalMapping = Video::getNormalMappingEnabled(),
       usingTextures      = Video::getTexturesEnabled();

  // activate texture and normal map texture
  if ( texID >= 0 )
    Video::useTexture(texID);
  else if ( true == usingTextures )
    Video::disableTextures();

  if ( normalMapID >= 0 )
    Video::useNormalMap(normalMapID);
  else if ( true == usingNormalMapping )
    Video::disableNormalMapping();

  Video::setModelTranslation(pos);
  Video::setModelRotation(rot);

  Video::applyTransforms();

  if ( mesh->getUnindexedVerts() != NULL )
    Video::renderTris( mesh->getUnindexedVerts(), mesh->getNumTris() );
  else
    Video::renderIndexedTris( mesh->getIndexedVerts(),
                              mesh->getIndices(),
                              mesh->getNumTris() );

  // re-enable textures if necessary
  if ( texID < 0 && true == usingTextures )
    Video::enableTextures();

  // re-enable normal mapping if necessary
  if ( normalMapID < 0 && true == usingNormalMapping )
    Video::enableNormalMapping();
}


void TriangleMeshEntity::update(float dt, BSP &world) {

}
